<?php
/**
 * Class that handles whitelabel functionality.
 *
 * @link    https://wpmudev.com
 * @since   4.11.1
 * @author  Joel James <joel@incsub.com>
 * @package WPMUDEV_Dashboard_Whitelabel
 */

// If this file is called directly, abort.
defined( 'WPINC' ) || die;

/**
 * Class WPMUDEV_Dashboard_Whitelabel
 */
class WPMUDEV_Dashboard_Whitelabel {

	/**
	 * WPMUDEV_Dashboard_Whitelabel constructor.
	 *
	 * @since 4.11.1
	 */
	public function __construct() {
		// Modify admin menus.
		add_action( 'admin_menu', array( $this, 'whitelabel_menu' ), 99 );
		add_action( 'network_admin_menu', array( $this, 'whitelabel_menu' ), 99 );

		// Remove deleted blogs.
		add_action( 'delete_blog', array( $this, 'remove_deleted_site' ) );
	}

	/**
	 * Print menu icon style inline.
	 *
	 * This is required because users may upload bigger sized
	 * icons even if we have recommended 20x20 size.
	 *
	 * @since 4.11.1
	 *
	 * @return void
	 */
	public function print_icon_style() {
		?>
		<style>
			#adminmenuwrap #adminmenu div.wp-menu-image img {
				width: 20px;
				height: 20px;
				object-fit: scale-down;
			}
		</style>
		<?php
	}

	/**
	 * White label admin menu for plugins.
	 *
	 * For WPMUDEV plugins, see if white label is configured
	 * in Dash. If so, change label and icon.
	 *
	 * @since 4.11.1
	 *
	 * @return void
	 */
	public function whitelabel_menu() {
		global $menu;

		// Get available configs.
		$configs = $this->get_menu_configs();
		// No need to continue if empty.
		if ( empty( $configs ) ) {
			return;
		}

		// Get plugin slugs.
		$enabled = array_keys( $configs );

		// On multisite, always do it on main site.
		if ( is_multisite() ) {
			// Doing this here to avoid multiple switches.
			switch_to_blog( $this->main_site_id() );
		}

		foreach ( $menu as $position => $data ) {
			// Only when it's required menu.
			if ( isset( $data[2] ) && in_array( $data[2], $enabled, true ) ) {
				// Change plugin menu title and icon if required.
				// phpcs:ignore
				$menu[ $position ] = $this->process_menu( $data );
			}
		}

		// Restore previous blog.
		if ( is_multisite() ) {
			restore_current_blog();
		}
	}

	/**
	 * On site delete, remove it from whitelabel list.
	 *
	 * Doing this to avoid unwanted errors if one site
	 * is dropped.
	 *
	 * @param int $site_id Site ID.
	 *
	 * @since 4.11.1
	 */
	public function remove_deleted_site( $site_id ) {
		// Get whitelabel settings.
		$settings = WPMUDEV_Dashboard::$site->get_whitelabel_settings();
		// Get already added sites.
		$sites = empty( $settings['labels_subsites'] ) ? array() : (array) $settings['labels_subsites'];
		// Delete current item if already exist.
		if ( in_array( $site_id, $sites ) ) { // phpcs:ignore
			// Remove the site id.
			$key = array_search( $site_id, $sites ); // phpcs:ignore
			unset( $sites[ $key ] );
		}
		// Update new list.
		WPMUDEV_Dashboard::$site->set_option( 'whitelabel_labels_subsites', $sites );
	}

	/**
	 * Process menu item and replace label and icon.
	 *
	 * When required, replace menu label and icon based
	 * on the configurations in white label settings.
	 *
	 * @param array $data Menu data.
	 *
	 * @since 4.11.1
	 *
	 * @return array $data
	 */
	private function process_menu( $data ) {
		// Get available configurations.
		$configs = $this->get_menu_configs();

		// Continue only if config for menu found.
		if ( isset( $configs[ $data[2] ] ) ) {
			$config = $configs[ $data[2] ];

			// If label changed.
			if ( ! empty( $config['name'] ) ) {
				// Replace menu label.
				$data[0] = $config['name'];
				// Rename page title.
				$data[3] = $config['name'];
			}

			// Replace menu icon.
			if ( isset( $data[6] ) ) {
				if ( ! empty( $config['icon_type'] ) ) {
					$print_style = false;
					switch ( $config['icon_type'] ) {
						case 'dashicon':
							// Set dashicons class.
							$class = sanitize_html_class( $config['icon_class'] );
							if ( ! empty( $class ) ) {
								$data[6] = 'dashicons-' . $class;
							}
							break;
						case 'upload':
							// Make sure the ID is int.
							$thumbnail_id = (int) $config['thumb_id'];
							// Get thumbnail url.
							$thumbnail_url = wp_get_attachment_image_url( $thumbnail_id, 'thumbnail', true );
							if ( ! empty( $thumbnail_id ) && ! empty( $thumbnail_url ) ) {
								$print_style = true;
								$data[6]     = $thumbnail_url;
							}
							break;
						case 'link':
							// Set direct link.
							$icon = esc_url( $config['icon_url'] );
							if ( ! empty( $icon ) ) {
								$print_style = true;
								$data[6]     = $icon;
							}
							break;
						case 'none':
							// No icon.
							$data[6] = 'none';
							break;
					}

					// Add a small inline style to fix icon size.
					if ( $print_style && ! has_action( 'admin_print_styles', 'print_icon_style' ) ) {
						add_action( 'admin_print_styles', array( $this, 'print_icon_style' ), 99 );
					}
				}
			}
		}

		/**
		 * Filter to modify plugin menu data.
		 *
		 * @param array $data Menu data.
		 *
		 * @since 4.11.1
		 */
		return apply_filters( 'wpmudev_dashboard_whitelabel_process_menu', $data );
	}

	/**
	 * Get all available plugin config data.
	 *
	 * Config data is prepared only for the active plugins.
	 * If any of the plugins change their slug if feature,
	 * the whitelabel won't work for that plugin.
	 *
	 * @since 4.11.1
	 *
	 * @return array
	 */
	private function get_menu_configs() {
		static $menu_configs = null;

		if ( null === $menu_configs ) {
			$menu_configs = array();

			// Get settings.
			$settings = WPMUDEV_Dashboard::$site->get_whitelabel_settings();

			// Check network settings.
			if ( is_multisite() && ! is_network_admin() && empty( $settings['labels_networkwide'] ) ) {
				// Site IDs.
				$sites = (array) $settings['labels_subsites'];

				// If current subsite is not selected.
				if ( empty( $sites ) || ! in_array( get_current_blog_id(), $sites )  ) {
					// Do not continue if no subsites selected.
					return apply_filters( 'wpmudev_dashboard_whitelabel_menu_configs', $menu_configs );
				}
			}

			// Only if enabled.
			if ( $settings['enabled'] && $settings['labels_enabled'] ) {
				// Get configuration.
				$config = (array) $settings['labels_config'];
				// WPMUDEV plugins ID and slug.
				$slugs = $this->get_plugin_slugs();

				// Go through each plugins.
				foreach ( $config as $pid => $data ) {
					// Make sure the ID is int.
					$pid = (int) $pid;
					// Include only if a valid plugin.
					if ( isset( $slugs[ $pid ] ) ) {
						$menu_configs[ $slugs[ $pid ] ] = $data;
					}
				}
			}
		}

		/**
		 * Filter to modify plugin menu config.
		 *
		 * @param array $plugins Plugins data.
		 *
		 * @since 4.11.1
		 */
		return apply_filters( 'wpmudev_dashboard_whitelabel_menu_configs', $menu_configs );
	}

	/**
	 * Get WPMUDEV plugin IDs and main menu slugs.
	 *
	 * @since 4.11.1
	 *
	 * @return mixed|void
	 */
	private function get_plugin_slugs() {
		// WPMUDEV plugins ID and slug.
		$slugs = array(
			51      => 'beehive',
			119     => 'wpmudev',
			167     => 'wds_wizard',
			248     => 'wpmudev-videos',
			9135    => 'branding',
			912164  => 'smush',
			1081721 => 'wphb',
			1081723 => 'wp-defender',
			1107020 => 'hustle',
			2097296 => 'forminator',
			2175128 => 'shipper',
			3760011 => 'snapshot',
			3779636 => 'wpmudev-hub',
		);

		/**
		 * Filter to modify plugin slugs list.
		 *
		 * @param array $slugs Slugs.
		 *
		 * @since 4.11.1
		 */
		return apply_filters( 'wpmudev_dashboard_whitelabel_plugin_slugs', $slugs );
	}

	/**
	 * Wrapper function to get main site ID.
	 *
	 * Function get_main_site_id() is introduced in 4.9.0. So we need
	 * to make sure we get the current id even in old versions.
	 *
	 * @since 4.11.1
	 *
	 * @return int
	 */
	private function main_site_id() {
		// Use get_main_site_id if it is available.
		if ( function_exists( 'get_main_site_id' ) ) {
			return get_main_site_id();
		}

		// If not multisite, return current ID.
		if ( ! is_multisite() ) {
			return get_current_blog_id();
		}

		// Get the network.
		$network = get_network();

		return $network ? $network->site_id : 0;
	}
}