<?php

/**
 * Class dealing with generating the GDPR Reports.
 *
 * @link       https://gdprdemo.willowdigital.com/
 * @since      1.0.0
 *
 * @package    Gdpr_Personal_Data_Reports
 * @subpackage Gdpr_Personal_Data_Reports/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Gdpr_Personal_Data_Reports
 * @subpackage Gdpr_Personal_Data_Reports/public
 * @author     Wojtek Murawski <wojtek@willowsconsulting.ie>
 */
class Gdpr_Personal_Data_Reports_Generator {

  /*
   * Statuses:
   * 0 - unconfirmed
   * 1 - confirmed
   * 2 - emailed
   * 3 - completed
   *
   * Request types:
   * 1 - GDPR data report request
   * 2 - GDPR account delete request
   */

  private $wpdb;
  private $table_name;
  public $status_unconfirmed = 0;
  public $status_confirmed = 1;
  public $status_emailed = 2;
  public $status_completed = 3;
  public function __construct($gdpr_personal_data_reports)
  {
    global $wpdb;
    $this->wpdb = $wpdb;
    $this->table_name = $this->wpdb->prefix . 'gdpr_requests';
    $this->gdpr_personal_data_reports = $gdpr_personal_data_reports;
  }

  /**
   * Store GDPR request in the database
   * @param [array] $data
   */
  public function add_request($data)
  {
    $query = $this->wpdb->prepare("INSERT INTO {$this->table_name} SET email = %s, status = %d, http_user_agent = %s, http_accept_language = %s, server_addr = %s, remote_addr = %s, confirmation_string = %s, request_time = %s, request_type = %d, timestamp = %d", array ( $data['email'], '0', $data['http_user_agent'], $data['http_accept_language'], $data['server_addr'], $data['remote_addr'], $data['confirmation_string'], $data['request_time'], $data['request_type'], $data['timestamp'] ) );
    $this->wpdb->query( $query );

    return $this->wpdb->insert_id;
  }

  /**
   * Create a new GDPR request
   * @param  [string] $email
   * @param  [array] $server
   */
  public function create_new_request($email, $server, $type, $url)
  {
    // Generate unique hash/code
    $confirmation_string = $this->generate_random_string(30);

    // Record the request
    $request_data = array(
      'confirmation_attempts' => 0,
      'confirmation_string' => $confirmation_string,
      'email' => $email,
      'code' => '',
      'http_user_agent' => $server['HTTP_USER_AGENT'],
      'http_accept_language' => $server['HTTP_ACCEPT_LANGUAGE'],
      'server_addr' => $server['SERVER_ADDR'],
      'remote_addr' => $server['REMOTE_ADDR'],
      'timestamp' => $server['REQUEST_TIME'],
      'request_time' => date("Y-m-d H:i:s", $server['REQUEST_TIME']),
      'request_type' => $type,
    );

    $request_id = $this->add_request($request_data);

    // Generate request code
    $request_code = $this->get_request_code($request_id, $server['REQUEST_TIME']);

    // Store request code
    $this->update_request_code($request_id, $request_code);

    // Send email with a link (confirm_gdpr?email=$email&code=$code)
    $this->email_confirmation_link($email, $request_code, $confirmation_string, $type, $url);

  }

  /**
   * Email the GDPR request confirmation link
   * @param  [string] $email
   * @param  [string] $request_code
   * @param  [string] $confirmation_string
   */
  public function email_confirmation_link($email, $request_code, $confirmation_string, $type, $url)
  {
    $site_name = get_option('blogname');

    $texts = array(
      'gdpr_report' => array(
        // GDPR Report
        'subject_text' => __('Please confirm your GDPR Request', $this->gdpr_personal_data_reports),
        'site_name' => $site_name,
        'confirmation_email_text_heading' => __('Confirm Your GDPR Request', $this->gdpr_personal_data_reports),
        'confirmation_email_text_instructions' => __('If you requested your personal data report from us please click the link below to confirm your email address and proceed with the request. If you DID NOT request it, please ignore this email.', $this->gdpr_personal_data_reports),
        'confirmation_email_text_confirm' => __('Confirm', $this->gdpr_personal_data_reports),
        'confirmation_email_text_smallprint' => __('We will generate and email you your personal data report as soon as possible. Usually it should be within 30 minutes after you confirm your request. If have not received your report within 24 hours from your confirmation, please try a new request or contact our customer support.', $this->gdpr_personal_data_reports),
        'confirmation_email_link' => $url . '?confirm_gdpr_request=1&request=' . $request_code . '&confirmation_code=' . $confirmation_string,
      ),
      // GDPR Account Deletion
      'gdpr_delete' => array(
        'subject_text' => __('Please confirm your ACCOUNT DELETION request', $this->gdpr_personal_data_reports),
        'site_name' => $site_name,
        'confirmation_email_text_heading' => __('Confirm Your ACCOUNT DELETION Request', $this->gdpr_personal_data_reports),
        'confirmation_email_text_instructions' => __('If you requested DELETING YOUR ACCOUNT on our website please click the link below to confirm your email address and proceed with the request. If you DID NOT request it, please ignore this email.', $this->gdpr_personal_data_reports),
        'confirmation_email_text_confirm' => __('Confirm', $this->gdpr_personal_data_reports),
        'confirmation_email_text_smallprint' => __('We will delete your personal data as soon as possible. Usually it should be completed within 30 minutes after you confirm your request. Due to the nature of this operation we cannot send you a confirmation email. Please bear in mind that some of your order related personal data may remain in our database a we are obliged to store it by tax regulations.', $this->gdpr_personal_data_reports),
        'confirmation_email_link' => $url . '?confirm_forget_me_request=1&request=' . $request_code . '&confirmation_code=' . $confirmation_string,
        'warning_text' => __('IMPORTANT! Clicking the link below will result in your account data being anonymized and deleted. This means you will no longer be able to login to your account and access your account information like order history. This operation CAN NOT BE REVERSED!', $this->gdpr_personal_data_reports),
      ),
    );

    // Build the template
    $template = new Gdpr_Personal_Data_Reports_Email();

    // GDPR Report request
    if($type == 1)
    {
      $subject = $texts['gdpr_report']['subject_text'];
      $body = $template->get_confirmation_body($texts['gdpr_report']);
    }

    // GDPR Deletion request
    if($type == 2)
    {
      $subject = $texts['gdpr_delete']['subject_text'];
      $body = $template->get_delete_confirmation_body($texts['gdpr_delete']);
    }

    $attachments = array();
    $headers = array('Content-Type: text/html; charset=UTF-8');

    // Send confirmation email
    if ( class_exists( 'WooCommerce' ) ) {
      wc_mail( $email, $subject, $body, $headers, $attachments );
    } else {
      wp_mail( $email, $subject, $body, $headers, $attachments );
    }

  }

  /**
   * Email personal data report to the user.
   * @param  [string] $data
   * @return
   */
  public function email_personal_data_report($body, $request_data)
  {
    $attachments = array();
    $email = $request_data->email;
    $subject = __('Your Personal Data Report', $this->gdpr_personal_data_reports);
    $headers = array('Content-Type: text/html; charset=UTF-8');

    if ( class_exists( 'WooCommerce' ) ) {
      wc_mail($email, $subject, $body, $headers, $attachments);
    } else {
      wp_mail( $email, $subject, $body, $headers, $attachments );
    }

  }

  /**
   * Generate personal data report
   * @param  [string] $request_code
   * @return [array]  $personal_data
   */
  public function generate_personal_data_report($request_code)
  {
    $personal_data = array();

    // Get User email
    $email = $this->get_request_email($request_code);
    // Get User ID
    $user = get_user_by( 'email', $email );
    // Get User data
    $user_data = get_userdata($user->ID);
    $user_meta = get_user_meta($user->ID);
    // Get User GDPR requests
    $gdpr_requests = $this->get_user_gdpr_requests($email);

    // Get User Comments
    $comments = get_comments( array('author_email' => $email));

    $settings = get_option($this->gdpr_personal_data_reports);

    $gdpr_request_statuses = array(
      '0' => __('Pending', $this->gdpr_personal_data_reports),
			'1' => __('Confirmed', $this->gdpr_personal_data_reports),
			'2' => __('Emailed', $this->gdpr_personal_data_reports),
			'3' => __('Account Deleted', $this->gdpr_personal_data_reports),
    );

    $gdpr_request_types = array(
      1 => __('Report', $this->gdpr_personal_data_reports),
      2 => __('Forget Me', $this->gdpr_personal_data_reports),
    );

    $personal_data = array(
      'ID' => $user->ID,
      'email' => $email,
      'comments' => $comments,
      'user_data' => $user_data,
      'user_meta' => $user_meta,
      'settings' => $settings,
      'gdpr_requests' => $gdpr_requests,
      'gdpr_requests_statuses' => $gdpr_request_statuses,
      'gdpr_requests_types' => $gdpr_request_types,
      'empty_record' => '---',
    );

    //echo('<pre>');
    //var_dump($personal_data);
    //die();

    return (object)$personal_data;
  }

  /**
   * Generate personal data report template
   * @param  [array] $data personal data array
   * @return [string] generated template
   */
  public function generate_personal_data_report_template($data)
  {
    $template = new Gdpr_Personal_Data_Reports_Email();

    return $template->get_report_template($data);
  }

  /**
   * Generate random string
   * @param  integer $length
   * @return string
  */
  public function generate_random_string($length = 30)
  {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charactersLength = strlen($characters);
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, $charactersLength - 1)];
    }
    return strtoupper($randomString);
  }

  /**
   * Get confirmation string of a request with a given request code
   * @param  [string] $request_code [request_id + substr(timestamp * 7, 0, 12) + substr(request_id * 6, 0, 5)]
   * @return [array]
   */
  public function get_confirmation_string($request_code)
  {
    $query = $this->wpdb->prepare( "SELECT request_id, email, confirmation_string FROM {$this->table_name} WHERE code = %s", array( $request_code ));
    $row = $this->wpdb->get_row( $query );

    return $row;
  }

  /**
   * Get the status of the request
   * @param  [type] $request_vars [description]
   * @return [type]               [description]
   */
  public function get_gdpr_request_status($request_vars)
  {
    $query = $this->wpdb->prepare( "SELECT status FROM {$this->table_name} WHERE code = %s", array( $request_vars['request']) );
    $row = $this->wpdb->get_row( $query );

    return (int)$row->status;
  }

  /**
   * Get a request code generated with request_id and timestamp
   * @param  [int] $request_id
   * @param  [int] $timestamp  [timestamp of the request]
   * @return [string]
   */
  public function get_request_code($request_id, $timestamp)
  {
    $request_code = $request_id . substr($timestamp * 7, 0, 12) . substr($request_id * 6, 0, 5);

    return $request_code;
  }

  /**
   * Get email address associated with the request
   * @param  [string] $request_code
   * @return [string]
   */
  public function get_request_email($request_code)
  {
    $query = $this->wpdb->prepare( "SELECT email FROM {$this->table_name} WHERE code = %s", array( $request_code ) );
    $row = $this->wpdb->get_row( $query );

    return $row->email;
  }

  /**
   * Get all user GDPR requests by email
   * @param  [string] $email
   * @return [array]
   */
  public function get_user_gdpr_requests($email)
  {
    $select = $this->wpdb->prepare( "SELECT * FROM {$this->table_name} WHERE email = %s", array( $email ) );
    $gdpr_requests = $this->wpdb->get_results($select);

    return $gdpr_requests;
  }

  /**
  * Returns all unique meta key from user meta database
  *
  * @param no parameter right now
  * @return std Class
  */
  public function get_user_meta_key()
  {
    $select = "SELECT distinct meta_key FROM " . $this->wpdb->usermeta;
    $usermeta = $this->wpdb->get_results($select);

    return $usermeta;
  }

  /**
   * Check if confirmation attempt is permitted.
   * @param  [array]  $query_vars
   * @return boolean
   */
  public function is_allowed_confirmation_attempt($request_code)
  {
    // Get max allowed attempts
    $gdpr_settings = get_option('gdpr-personal-data-reports');
    $max_confirmation_attempts = (int)$gdpr_settings['max_confirmation_attempts'];

    // Get number of request done in the last 24h
    $query = $this->wpdb->prepare( "SELECT confirmation_attempts FROM {$this->table_name} WHERE code = %s", array( $request_code ) );
    $row = $this->wpdb->get_row( $query );

    $confirmation_attempts = $row->confirmation_attempts;

    if($confirmation_attempts >= $max_confirmation_attempts) {
      return false;
    } else {
      return true;
    }
  }

  /**
	 * Check if customer is allowed to make a GDPR request and did not exceed the
	 * maximum of daily requests permitted.
	 * @param  [string]  $email
	 * @return boolean
	 */
  public function is_allowed_gdpr_request($email)
  {
    // Get max allowed requests per 24h
    $gdpr_settings = get_option('gdpr-personal-data-reports');
    $max_allowed = (int)$gdpr_settings['max_requests'];

    // Get number of request done in the last 24h
    $query = $this->wpdb->prepare( "SELECT COUNT(request_id) as requests_made FROM {$this->table_name} WHERE email = %s AND request_time > NOW() - INTERVAL 1 DAY", array( $email ) );
    $row = $this->wpdb->get_row( $query );

    $requests_made = $row->requests_made;

    if($requests_made >= $max_allowed) {
      return false;
    } else {
      return true;
    }
  }

  /**
   * Validate the email input. Check if format is correct, the email exists
   * in the database and the user did not exceed daily request limit.
   * @param  [type] $email [description]
   * @return [type]        [description]
   */
  public function validate_email($email)
  {
    $errors = array();

    // Email format
    if (!preg_match('/^[^\@]+@.*.[a-z]{2,15}$/i', $email)) {
      $errors['email'] = __('Your email address seems to have incorrect format.', $this->gdpr_personal_data_reports);
      $bad_email_format = true;
    }

    // Email exists in the database
    if(!email_exists($email) && empty($bad_email_format)) {
      $errors['email'] = __('There is no account associated with this email address in our system', $this->gdpr_personal_data_reports);
    }

    // Customer did not exceed the daily limit of requests
    if(!$this->is_allowed_gdpr_request($email)) {
      $errors['email'] = __('You have exceeded the daily limit of GDPR requests, please try again tomorrow', $this->gdpr_personal_data_reports);
    }

    return $errors;
  }

  /**
   * Update the Request Code
   * @param  [int] $request_id
   * @param  [string] $request_code
   */
  public function update_request_code($request_id, $request_code)
  {
    $query = $this->wpdb->prepare( "UPDATE {$this->table_name} SET code = %s WHERE request_id = %d", array( $request_code, $request_id ) );

    $this->wpdb->query( $query );
  }

  /**
   * Update the Request Status
   * @param  [string] $request_code
   * @param  [int] $status
   */
  public function update_gdpr_request_status($request_code, $status)
  {
    $query = $this->wpdb->prepare( "UPDATE {$this->table_name} SET status = %s WHERE code = %s", array( $status, $request_code ) );
    $this->wpdb->query( $query );
  }

  /**
   * Update the Request Status
   * @param  [string] $request_code
   * @param  [int] $status
   */
  public function update_gdpr_request_confirmation_attempts($request_code)
  {
    $query = $this->wpdb->prepare("UPDATE {$this->table_name} SET confirmation_attempts = (confirmation_attempts + 1) WHERE code = %s", array( $request_code ) );
    $this->wpdb->query( $query );
  }

  /**
   * [verify_gdpr_request description]
   * @param  [type] $get_request [description]
   * @return [type]              [description]
   */
  public function verify_gdpr_request($request_vars)
  {
    $confirmation = $this->get_confirmation_string($request_vars['request']);
    $confirmation_string = $confirmation->confirmation_string;

    // Add confirmation attempt
    $this->update_gdpr_request_confirmation_attempts($request_vars['request']);

    // Check if confirmation string in the request match what is stored in the db
    if($confirmation_string !== $request_vars['confirmation_code'])
    {
      return false;
    }

    return $confirmation;

  }

  /**
   * [load_template_part description]
   * @param  [type] $template_name [description]
   * @param  [type] $part_name     [description]
   * @return [type]                [description]
   */
  public function load_template_part($template_name, $part_name=null) {
    ob_start();
    get_template_part($template_name, $part_name);
    $var = ob_get_contents();
    ob_end_clean();
    return $var;
}

}
