<?php

namespace Beehive\GuzzleHttp\Cookie;

use Beehive\GuzzleHttp\Utils;
/**
 * Persists non-session cookies using a JSON formatted file
 */
class FileCookieJar extends \Beehive\GuzzleHttp\Cookie\CookieJar
{
    /** @var string filename */
    private $filename;
    /**
     * Create a new FileCookieJar object
     *
     * @param string $cookieFile File to store the cookie data
     *
     * @throws \RuntimeException if the file cannot be found or created
     */
    public function __construct($cookieFile)
    {
        $this->filename = $cookieFile;
        if (\file_exists($cookieFile)) {
            $this->load($cookieFile);
        }
    }
    /**
     * Saves the file when shutting down
     */
    public function __destruct()
    {
        $this->save($this->filename);
    }
    /**
     * Saves the cookies to a file.
     *
     * @param string $filename File to save
     * @throws \RuntimeException if the file cannot be found or created
     */
    public function save($filename)
    {
        $json = [];
        foreach ($this as $cookie) {
            if ($cookie->getExpires() && !$cookie->getDiscard()) {
                $json[] = $cookie->toArray();
            }
        }
        if (\false === \file_put_contents($filename, \json_encode($json), \LOCK_EX)) {
            // @codeCoverageIgnoreStart
            throw new \RuntimeException("Unable to save file {$filename}");
            // @codeCoverageIgnoreEnd
        }
    }
    /**
     * Load cookies from a JSON formatted file.
     *
     * Old cookies are kept unless overwritten by newly loaded ones.
     *
     * @param string $filename Cookie file to load.
     * @throws \RuntimeException if the file cannot be loaded.
     */
    public function load($filename)
    {
        $json = \file_get_contents($filename);
        if (\false === $json) {
            // @codeCoverageIgnoreStart
            throw new \RuntimeException("Unable to load file {$filename}");
            // @codeCoverageIgnoreEnd
        }
        $data = \Beehive\GuzzleHttp\Utils::jsonDecode($json, \true);
        if (\is_array($data)) {
            foreach (\Beehive\GuzzleHttp\Utils::jsonDecode($json, \true) as $cookie) {
                $this->setCookie(new \Beehive\GuzzleHttp\Cookie\SetCookie($cookie));
            }
        } elseif (\strlen($data)) {
            throw new \RuntimeException("Invalid cookie file: {$filename}");
        }
    }
}