<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\Widgets
 * @author  StudioPress
 * @license GPL-2.0-or-later
 * @link    https://my.studiopress.com/themes/genesis/
 */

add_action( 'widgets_init', 'genesis_handle_block_widget_editor' );
/**
 * Default the block-based widget editor to be disabled, which is defaulted to be enabled in WP 5.8.
 *
 * @since 3.3.4
 */
function genesis_handle_block_widget_editor() {
	// Disable the widgets block editor feature in core.
	add_filter( 'use_widgets_block_editor', '__return_false', 0 );
}

add_action( 'admin_init', 'genesis_handle_dismissing_block_widget_editor_notice' );
/**
 * Handle dismissing the admin notice for the block widget editor optin.
 *
 * @since 3.3.4
 */
function genesis_handle_dismissing_block_widget_editor_notice() {
	// Only handle dismissing the block widget editor notice if "?genesis_widget_block_editor_dismiss=1" is in the URL.
	if ( filter_input( INPUT_GET, 'genesis_widget_block_editor_dismiss', FILTER_SANITIZE_STRING ) !== '1' ) {
		// Exit early.
		return;
	}

	// If no nonce exists in the URL, do nothing.
	if ( ! isset( $_GET['_wpnonce'] ) ) {
		return;
	}

	// If the nonce fails, do nothing.
	if ( ! wp_verify_nonce( filter_input( INPUT_GET, '_wpnonce', FILTER_SANITIZE_STRING ), 'genesis-widget-block-editor-dismiss' ) ) {
		return;
	}

	// Make sure the current user has permission to dismiss the notice.
	if ( ! current_user_can( 'administrator' ) ) {
		return;
	}

	// Set the notice to be dismissed.
	update_option( 'genesis_notice_dismissed_use_widgets_block_editor', true );

}


add_action( 'admin_notices', 'genesis_block_widgets_optin_notification' );
/**
 * Show an admin notice on the widgets screen informing the user how to opt-in to block widgets.
 *
 * @since 3.3.4
 */
function genesis_block_widgets_optin_notification() {

	$current_screen = get_current_screen();

	// Only show on the widgets screen.
	if ( 'widgets' !== $current_screen->id ) {
		return;
	}

	// Only the show the notice to an administrator.
	if ( ! current_user_can( 'administrator' ) ) {
		return;
	}

	// If this notice has been dismissed already, don't show it.
	if ( get_option( 'genesis_notice_dismissed_use_widgets_block_editor' ) ) {
		return;
	}

	// Get the WordPress version.
	require ABSPATH . WPINC . '/version.php'; // $wp_version;

	// If the WordPress version is older than 5.8, show no admin notice.
	if ( version_compare( $wp_version, '5.8-RC1-51325', '<' ) ) {
		return false;
	}

	?>
	<div class="notice notice-success">
		<p>
			<?php
			echo wp_kses_post(
				sprintf(
					// Translators: The link html around the help and dismiss links for the widget block editor opt-in.
					__( 'Note from Genesis: To maintain the best site editing experience for you, we\'ve disabled the widget screen introduced in WordPress 5.8. To learn more about the 5.8 widget experience and how to activate it, click the following link: %1$sLearn more%2$s %3$sDismiss%4$s', 'genesis' ),
					'<p><a href="https://my.studiopress.com/documentation/snippets/block-editor/enable-block-based-widget-editor" target="_blank" rel="noopener noreferrer" class="button">',
					'</a>',
					'<a href="' . wp_nonce_url( add_query_arg( [ 'genesis_widget_block_editor_dismiss' => 1 ], admin_url( 'widgets.php' ) ), 'genesis-widget-block-editor-dismiss' ) . '" rel="noopener noreferrer" class="button">',
					'</a></p>'
				)
			);
			?>
		</p>
	</div>
	<?php
}

add_action( 'widgets_init', 'genesis_load_widgets' );
/**
 * Register widgets for use in the Genesis theme.
 *
 * @since 1.7.0
 */
function genesis_load_widgets() {

	register_widget( 'Genesis_Featured_Page' );
	register_widget( 'Genesis_Featured_Post' );
	register_widget( 'Genesis_User_Profile_Widget' );

}

add_action( 'load-themes.php', 'genesis_remove_default_widgets_from_header_right' );
/**
 * Temporary function to work around the default widgets that get added to
 * Header Right when switching themes.
 *
 * The $defaults array contains a list of the IDs of the widgets that are added
 * to the first sidebar in a new default install. If this exactly matches the
 * widgets in Header Right after switching themes, then they are removed.
 *
 * This works around a perceived WP problem for new installs.
 *
 * If a user amends the list of widgets in the first sidebar before switching to
 * a Genesis child theme, then this function won't do anything.
 *
 * @since 1.8.0
 *
 * @return void Return early if not just switched to a new theme.
 */
function genesis_remove_default_widgets_from_header_right() {

	// Some tomfoolery for a faux activation hook.
	if ( ! isset( $_REQUEST['activated'] ) || 'true' !== $_REQUEST['activated'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- No data is being processed.
		return;
	}

	$widgets  = get_option( 'sidebars_widgets' );
	$defaults = [
		0 => 'search-2',
		1 => 'recent-posts-2',
		2 => 'recent-comments-2',
		3 => 'archives-2',
		4 => 'categories-2',
		5 => 'meta-2',
	];

	if ( isset( $widgets['header-right'] ) && $defaults === $widgets['header-right'] ) {
		$widgets['header-right'] = [];
		update_option( 'sidebars_widgets', $widgets );
	}

}
